<?php
/**
 * 
 * Abstract application controller
 * 
 * @author Renato Peterman <renato.pet@gmail.com>
 *
 */

namespace Rest\Controller;

use API\Exceptions\ApiBadRequestException;
use API\Exceptions\ApiException;
use API\Exceptions\ApiNotFoundException;
//use Symfony\Component\Serializer\Encoder\JsonEncoder;
//use Symfony\Component\Serializer\Normalizer\GetSetMethodNormalizer;
//use Symfony\Component\Serializer\Serializer;
//use Symfony\Component\Serializer\Encoder\XmlEncoder;
use Doctrine\ORM\QueryBuilder;
use Zend\Mvc\Controller\AbstractRestfulController;
use Zend\Http\Response\Stream;
use Zend\Http\Headers;
use DoctrineModule\Stdlib\Hydrator\DoctrineObject as DoctrineHydrator;
use Doctrine\ORM\Tools\Pagination\Paginator as ORMPaginator;
use DoctrineORMModule\Paginator\Adapter\DoctrinePaginator as DoctrineAdapter;
use Zend\Paginator\Paginator;

abstract class AbstractApplicationRestfulController extends AbstractRestfulController
{
    protected $em;
    protected $authManager;
    protected $whiteList = array('login'); # whitelisted routes

    /**
     * 
     * Verify if user was authenticated
     * 
     * @override
     */
    public function onDispatch(\Zend\Mvc\MvcEvent $e) {
        
        if(!$this->getAuthManager()->getIdentity() && !in_array($e->getRouteMatch()->getMatchedRouteName(), $this->whiteList)){
            return $this->redirect()->toRoute('login');
        }

        parent::onDispatch($e);

    }
    
    /**
     * 
     * Return a Doctrine Entity Manager instance
     * 
     * @return Doctrine\ORM\EntityManager
     */
    public function getEntityManager()
    {
        if (null === $this->em) {
            $this->em = $this->getServiceLocator()->get('Doctrine\ORM\EntityManager');
        }
        return $this->em;
    }
    
    /**
     * 
     * Return a Zend Authentication Service instance
     * 
     * @return Zend\Authentication\AuthenticationService
     */
    public function getAuthManager()
    {
        if (null === $this->authManager) {
            $this->authManager = $this->getServiceLocator()->get('Zend\Authentication\AuthenticationService');
        }
        return $this->authManager;
    }
    
    /**
     * 
     * Create a stream response
     * 
     * @param string $file
     * @param string $contentType
     * @return \Zend\Http\Response\Stream
     */
    public function createFileResponse($file, $contentType){
        $response = new Stream();
        $response->setStream(fopen($file, 'r'));
        $response->setStatusCode(200);
        $headers = new Headers();
        $headers->addHeaderLine('Content-Transfer-Encoding', 'binary')
                ->addHeaderLine('Content-Type', $contentType)
                ->addHeaderLine('Content-Length', filesize($file));

        $response->setHeaders($headers);
        return $response;
    }

    /**
     * Format the data that will be showed in view
     *
     * @param  array $data
     * @param  string $format
     * @return mixed
     */
    protected function createResponse($content, $httpStatusCode = 200, $error = false, $message = ''){

        $this->getResponseWithHeader()->setStatusCode($httpStatusCode);

        $data = null;
        if($error){
            $data = array('error' => $error, 'message' => $message);
        }else{
            $data = $content;
        }

        $format = $this->getEvent()->getRouteMatch()->getParam('format');

        // Datetime callback
        /*
        $datetimeCallback = function ($dateTime) {
            error_log($dateTime);
            return $dateTime instanceof \DateTime ? $dateTime->format(\DateTime::ISO8601) : '';
        };

        $encoders = array(new JsonEncoder(), new XmlEncoder());
        $normalizer = new GetSetMethodNormalizer();

        $normalizer->setCircularReferenceHandler(function ($object) {
            if($object){
                return $object->getId();
            }
        });

        $normalizer->setCallbacks(array('created' => $datetimeCallback, 'updated' => $datetimeCallback));
        $serializer = new Serializer(array($normalizer), $encoders);

        return $this->createCustomResponse( $serializer->serialize($data, $format) );
        */

        $serializer = $this->getServiceLocator()->get('jms_serializer.serializer');

        // Paginated
        if($content instanceof QueryBuilder){

            $limit = (int)$this->params()->fromQuery('limit', false);
            $page = (int)$this->params()->fromQuery('page', 0);
            if(!$limit){
                $results = $data->getQuery()->getResult();
            }else{

                $this->getResponseWithHeader()->setStatusCode(200);
                $format = $this->getEvent()->getRouteMatch()->getParam('format');

                // Sort
                $sort = $this->params()->fromQuery('sort', false);
                $sortOrder = $this->params()->fromQuery('sortOrder', "ASC");
                if($sort){
                    $data->orderBy("obj.{$sort}", $sortOrder);
                }

                $adapter = new DoctrineAdapter(new ORMPaginator($data));
                $paginator = new Paginator($adapter);

                $paginator->setDefaultItemCountPerPage($limit);

                if($page) $paginator->setCurrentPageNumber($page);

                $results = array();
                if($paginator->getTotalItemCount() > 0){
                    foreach($paginator as $obj){
                        $results[] = $obj;
                    }
                }

                $this->getResponse()->getHeaders()->addHeaderLine('Access-Control-Expose-Headers', 'X-Count-Items, X-Count-Pages');
                $this->getResponse()->getHeaders()->addHeaderLine('X-Count-Items', $paginator->getTotalItemCount());
                $this->getResponse()->getHeaders()->addHeaderLine('X-Count-Pages', $paginator->getPages()->pageCount);
            }

            return $this->createCustomResponse( $serializer->serialize($results, $format) );

        }else{
            return $this->createCustomResponse( $serializer->serialize($data, $format) );
        }

    }

    /**
     * Create a response for a generic exception
     * @param \Exception $ex
     * @return \Zend\View\Model\JsonModel
     */
    protected function createGenericApiExceptionResponse($ex){
        return $this->createErrorResponse($ex->getMessage(), 404);
    }

    /**
     * Create a response for a ApiException exception
     * @param \API\Exceptions\ApiException $ex
     * @return \Zend\View\Model\JsonModel
     */
    protected function createApiExceptionResponse(ApiException $ex){
        return $this->createErrorResponse($ex->getMessage(), $ex->getCode());
    }


    /**
     * Create a error response
     * @param \API\Exceptions\ApiException $ex
     * @return \Zend\View\Model\JsonModel
     */
    protected function createErrorResponse($message, $status = 500){
        return $this->createResponse(null, $status, true, $message); // 200 - Http Request OK
    }

    /**
     * Create a Bad Request error response (400)
     * @param \API\Exceptions\ApiException $ex
     * @return \Zend\View\Model\JsonModel
     */
    protected function createBadRequestResponse(){
        return $this->createResponse(null, 400, true, "Bad request");
    }

    /**
     * Create a Unauthorized error response (401)
     * @param \API\Exceptions\ApiException $ex
     * @return \Zend\View\Model\JsonModel
     */
    protected function createUnauthorizedResponse(){
        return $this->createResponse(null, 401, true, "Unauthorized request");
    }

    /**
     * Create a Forbidden error response (403)
     * @param \API\Exceptions\ApiException $ex
     * @return \Zend\View\Model\JsonModel
     */
    protected function createForbiddenResponse(){
        return $this->createResponse(null, 403, true, "Forbidden");
    }

    /**
     * Create a Not Found error response (404)
     * @param \API\Exceptions\ApiException $ex
     * @return \Zend\View\Model\JsonModel
     */
    protected function createNotFoundResponse(){
        return $this->createResponse(null, 404, true, "Resource not found");
    }

    /**
     * Create a Method Not Allowed error response (405)
     * @param \API\Exceptions\ApiException $ex
     * @return \Zend\View\Model\JsonModel
     */
    protected function createMethodNotAllowedResponse(){
        return $this->createResponse(null, 405, true, "Method not allowed");
    }

    /**
     * __get override
     * @param $name
     * @return string
     */
    public function __get($name) {
        if($name == "entityClass"){
            return $this->getEntityClass();
        }
    }

    /**
     * Return the controller default class name based on controller name
     * @return string
     */
    public function getEntityClass(){
        $parts = explode("\\", get_called_class());
        $name = str_replace("Controller", "", $parts[count($parts)-1]);
        return "Model\\Entity\\" . $name; 
    }

    /**
     * Redirect to default entity/controller route
     * @param string $action
     * @return \Zend\Http\Response
     */
    public function redirectToDefaultRoute($action = null){
        if(!$action){
            $action = "index";
        }
        $parts = explode("\\", get_called_class());
        $name = str_replace("Controller", "", $parts[count($parts)-1]);
        return $this->redirect()->toRoute("application/default", array("controller" => strtolower($name), "action" => $action));
    }

    /**
     * Create a custom controller response
     * @param $content
     * @return mixed
     */
    private function createCustomResponse($content){

        $format = $this->getEvent()->getRouteMatch()->getParam('format');
        $response = $this->getResponseWithHeader();
        if(!$format){
            $response->getHeaders()->addHeaderLine('Content-Type','application/json');
        }else{
            if($format == 'json'){
                $response->getHeaders()->addHeaderLine('Content-Type','application/json');
            }else if($format == 'xml'){
                $response->getHeaders()->addHeaderLine('Content-Type','application/xml');
            }
        }

        return $response->setContent($content);
    }

    /**
     * Get response object with headers
     * @return \Zend\Stdlib\ResponseInterface
     */
    private function getResponseWithHeader()
    {
        $response = $this->getResponse();
        $response->getHeaders()
            ->addHeaderLine('Access-Control-Allow-Origin','*')
            ->addHeaderLine('Access-Control-Allow-Methods','POST PUT DELETE GET');
        return $response;
    }

    /**
     * Generic Methods
     */

    /**
     * Get entity list
     * @return mixed|JsonModel
     */
    public function getList()
    {
        try{

            $qb = $this->getEntityManager()->createQueryBuilder()
                ->select('obj')
                ->from($this->getEntityClass(),'obj');

            $firstResult = $this->params()->fromQuery('offset',false);
            if($firstResult){
                $qb->setFirstResult($firstResult);
            }

            $maxResults = $this->params()->fromQuery('limit',false);
            if($maxResults){
                $qb->setMaxResults($maxResults);
            }

            return $this->createResponse($qb);
            // $data = $qb->getQuery()->getResult();

        }catch (ApiException $ex){

            return $this->createApiExceptionResponse($ex);

        }catch (\Exception $ex){

            return $this->createGenericApiExceptionResponse($ex);

        }

        return $this->createResponse($qb);
        // return $this->createResponse($data);
    }

    /**
     * Get entity
     * @param mixed $id
     * @return mixed|JsonModel
     */
    public function get($id)
    {
        try{

            $data = $this->getEntityManager()->createQueryBuilder()
                ->select('obj')
                ->from($this->getEntityClass(),'obj')
                ->where('obj.id = :objId')
                ->setParameter("objId",$id)
                ->getQuery()->getOneOrNullResult();
                //->getQuery()->getOneOrNullResult(\Doctrine\ORM\Query::HYDRATE_ARRAY);

        }catch (ApiException $ex){

            return $this->createApiExceptionResponse($ex);

        }catch (\Exception $ex){

            return $this->createGenericApiExceptionResponse($ex);

        }

        return $this->createResponse($data);
    }

    /**
     * Create and update entity
     * @param mixed $data
     * @return JsonModel
     * @throws \ApiBadRequestException
     */
    public function create($data)
    {
        try{

            if (!$data){
                throw new ApiBadRequestException('Nenhuma informação enviada');
            }

            $entity = array_key_exists('id', $data) && $data['id'] ? $this->getEntityManager()->find($this->getEntityClass(), $data['id']) : null;
            if(!$entity){
                $entity = new $this->entityClass;
            }

            // Remove dates
            unset($data['created']);
            unset($data['updated']);

            $hydrator = new DoctrineHydrator($this->getEntityManager(), $this->getEntityClass());
            $hydrator->hydrate($data, $entity);

            if($entity->getId()){
                $this->getEntityManager()->merge($entity);
            }else{
                $this->getEntityManager()->persist($entity);
            }
            $this->getEntityManager()->flush();

            return $this->createResponse( $entity );

        }catch (ApiException $ex){

            return $this->createApiExceptionResponse($ex);

        }catch (\Exception $ex) {

            return $this->createGenericApiExceptionResponse($ex);

        }
    }

    /**
     * Update entity (not used)
     * @param mixed $id
     * @param mixed $data
     * @return JsonModel
     * @throws \ApiBadRequestException
     */
    public function update($id, $data)
    {
        try{

            if (!$data){
                throw new ApiBadRequestException('Nenhuma informação enviada');
            }

            if (!$id){
                throw new ApiBadRequestException('Código inválido');
            }

            $entity = $this->getEntityManager()->find($this->getEntityClass(), $id);
            if(!$entity){
                throw new ApiNotFoundException('Registro não encontrado');
            }

            // Remove dates
            unset($data['created']);
            unset($data['updated']);

            $hydrator = new DoctrineHydrator($this->getEntityManager(), $this->getEntityClass());
            $hydrator->hydrate($data, $entity);

            $this->getEntityManager()->merge($entity);
            $this->getEntityManager()->flush();

            return $this->createResponse( $entity );

        }catch (ApiException $ex){

            return $this->createApiExceptionResponse($ex);

        }catch (\Exception $ex) {

            return $this->createGenericApiExceptionResponse($ex);

        }

    }

    /**
     * Delete entity
     * @param mixed $id
     * @return mixed|JsonModel
     */
    public function delete($id)
    {

        try{

            if (!$id){
                throw new ApiBadRequestException('Código inválido');
            }

            $entity = $this->getEntityManager()->find($this->getEntityClass(), $id);
            if(!$entity){
                throw new ApiNotFoundException('Register not found');
            }

            $this->getEntityManager()->remove($entity);
            $this->getEntityManager()->flush();

            return $this->createResponse(array(
                'success' => true
            ));

        }catch (ApiException $ex){

            return $this->createApiExceptionResponse($ex);

        }catch (\Exception $ex) {

            return $this->createGenericApiExceptionResponse($ex);

        }

    }
    
}
