<?php

namespace Model\Entity;

use Commons\Util\ImageUtil;
use Doctrine\ORM\Mapping as ORM;
use JMS\Serializer\Annotation as JMS;
use Zend\Form\Annotation;

/**
 * Model\Entity\Catalog
 *
 * @ORM\Entity()
 * @ORM\Table(name="catalog")
 * @ORM\HasLifecycleCallbacks
 */
class Catalog extends AbstractEntity
{
    const DEFAULT_PATH = './data/images/catalog';

    const TYPE_PDF = 'pdf';
    const TYPE_IMAGE = 'image';

    /**
     * @ORM\Id
     * @ORM\Column(type="integer")
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    protected $id;

    /**
     * @ORM\Column(type="string", length=255)
     */
    protected $description;

    /**
     * @ORM\Column(name="type", type="string", length=255)
     */
    protected $type;

    /**
     * @ORM\Column(name="thumb", type="string", length=255, nullable=true)
     */
    protected $thumb;

    /**
     * @ORM\Column(name="file", type="string", length=255, nullable=true)
     */
    protected $file;

    /**
     * @ORM\OneToMany(targetEntity="Model\Entity\CatalogImage", mappedBy="catalog", cascade={"persist", "merge", "remove"})
     * @ORM\JoinColumn(name="id", referencedColumnName="id_catalog")
     * @ORM\OrderBy({"order" = "ASC"})
     *
     * @JMS\Type("Model\Entity\CatalogImage")
     */
    protected $catalogImages;

    /**
     * @ORM\PreRemove()
     */
    public function preDelete() {
        if($this->getType() == static::TYPE_PDF){
            if(file_exists($this->getFilePath())){
                unlink($this->getFilePath());
            }
            if(file_exists($this->getThumbPath())){
                unlink($this->getThumbPath());
            }
        }else{
            if(count($this->getCatalogImages()) > 0){
                foreach($this->getCatalogImages() as $img){
                    if(file_exists($img->getFilePath())){
                        unlink($img->getFilePath());
                    }
                    if(file_exists($img->getThumbPath())){
                        unlink($img->getThumbPath());
                    }
                }
            }
        }
        if(file_exists($this->getFolder()) && is_dir($this->getFolder())){
            rmdir($this->getFolder());
        }
    }

    public function __construct()
    {
        parent::__construct();
    }

    /**
     * @return mixed
     */
    public function getFile()
    {
        return $this->file;
    }

    /**
     * @param mixed $file
     */
    public function setFile($file)
    {
        $this->file = $file;
    }

    /**
     * @return mixed
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * @param mixed $type
     */
    public function setType($type)
    {
        $this->type = $type;
    }

    /**
     * @return mixed
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * @param mixed $description
     */
    public function setDescription($description)
    {
        $this->description = $description;
    }

    /**
     * @return mixed
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @param mixed $id
     */
    public function setId($id)
    {
        $this->id = $id;
    }

    /**
     * @return mixed
     */
    public function getCatalogImages()
    {
        return $this->catalogImages;
    }

    /**
     * @param mixed $catalogImages
     */
    public function setCatalogImages($catalogImages)
    {
        $this->catalogImages = $catalogImages;
    }

    /**
     * @return mixed
     */
    public function getThumb()
    {
        return $this->thumb;
    }

    /**
     * @param mixed $thumb
     */
    public function setThumb($thumb)
    {
        $this->thumb = $thumb;
    }

    public function __toString() {
        return $this->getDescription();
    }

    /**
     * Get path for file and id
     * @param $file
     * @param $id
     * @return string
     */
    public static function getPathForFile($file = '', $catalogId = ''){
        $folder = self::DEFAULT_PATH . '/' . $catalogId;;
        if(!file_exists($folder)){
            mkdir($folder, 0777, true);
        }
        return $folder . '/' . $file;
    }

    /**
     * Get path for file and id
     * @param $file
     * @param $id
     * @return string
     */
    public function getFilePath(){
        return Catalog::getPathForFile($this->getFile(), $this->getId());
    }

    /**
     * Get path for thumb and id
     * @param $file
     * @param $id
     * @return string
     */
    public function getThumbPath(){
        return Catalog::getPathForFile($this->getThumb(), $this->getId());
    }

    /**
     * Get default folder
     * @return string
     */
    public function getFolder(){
        return self::DEFAULT_PATH . '/' . $this->getId();
    }

    public function setFileResource($file){

        if(!$file){
            throw new \Exception("Arquivo inválido");
        }

        $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $hash = md5(uniqid(rand(), true));
        $filename = 'pdf_' . $hash  . '.' . $extension;
        $filenameThumb = 'pdf_' . $hash  . '_thumb.jpg'; //Fixed JPG file
        $filePath = \Model\Entity\Catalog::getPathForFile($filename, $this->getId());
        $thumbPath = \Model\Entity\Catalog::getPathForFile($filenameThumb, $this->getId());

        move_uploaded_file($file['tmp_name'], $filePath);
        $this->setFile($filename);

        // Generate thumb
        if(\Commons\Util\GhostScriptUtil::isAvailable()){
            \Commons\Util\GhostScriptUtil::generatePDFPreview($filePath, $thumbPath);
            if(file_exists($thumbPath)){
                ImageUtil::resize($thumbPath, $thumbPath, "jpg", 160, 240, 50);
                $this->setThumb($filenameThumb);
            }
        }

    }
}