<?php
/**
 * Order controller
 *
 * @author Renato Peterman <renato.pet@gmail.com>
 */

namespace Application\Controller;

use Model\Entity\OrderProductVariant;
use Zend\View\Model\JsonModel;
use Zend\View\Model\ViewModel;
use DoctrineModule\Stdlib\Hydrator\DoctrineObject as DoctrineHydrator;

class OrderController extends AbstractApplicationController
{

    public function indexAction()
    {
        $config = $this->getDefaultConfig();
        $hasRepresentedCompanies = $this->getEntityManager()->createQuery("SELECT COUNT(obj) FROM Model\\Entity\\RepresentedCompany obj")->getSingleScalarResult();
        return new ViewModel(array(
            'configuration' => array(
                'enableUpdateFromClient' => $config->getEnableOrderUpdateFromClientFunctionality()
            ),
            'hasRepresentedCompanies' => $hasRepresentedCompanies
        ));
    }

    public function viewAction()
    {
        $id = $this->params()->fromRoute('id', false);

        $entity = null;
        if($id) {
            $entity = $this->getEntityManager()->find('Model\Entity\Order', $id);
        }

        $hasRepresentedCompanies = $this->getEntityManager()->createQuery('SELECT COUNT(obj) FROM Model\Entity\RepresentedCompany obj')->getSingleScalarResult();

        return new ViewModel(array(
            'entity' => $entity,
            'hasRepresentedCompanies' => $hasRepresentedCompanies
        ));
    }

    public function getProductInfoAction(){

        $retorno = array('error' => true, 'message' => '');

        try{

            $productId = $this->params()->fromPost('product', false);
            if(!$productId){
                throw new \Exception('Código do produto inválido');
            }

            $product = $this->getEntityManager()->find('Model\Entity\Product', $productId);
            if(!$product){
                throw new \Exception('Produto não encontrado');
            }

            $warehouseId = $this->params()->fromPost('warehouse', false);
            if(!$warehouseId){
                throw new \Exception('Código do estoque inválido');
            }

            $warehouse = $this->getEntityManager()->find('Model\Entity\Warehouse', $warehouseId);
            if(!$warehouse){
                throw new \Exception('Estoque não encontrado');
            }

            $pw = $this->getEntityManager()->find('Model\Entity\ProductWarehouse', array('product' => $product, 'warehouse' => $warehouse));
            if(!$pw){
                throw new \Exception('Estoque de produto não encontrado');
            }

            $arr = array();
            $arr['id'] = $product->getId();
            $arr['name'] = $product->getName();
            $arr['price'] = $product->getPriceCurrency();;
            $arr['stock'] = $pw->getAmountRound();

            return new \Zend\View\Model\JsonModel($arr);

        }catch(\Exception $ex){
            $retorno['message'] = $ex->getMessage();
        }

        return new \Zend\View\Model\JsonModel($retorno);

    }

    public function getProductsAction(){

        $retorno = array('error' => true, 'message' => '');

        try{

            $id = $this->params()->fromRoute('id',false);
            if(!$id){
                throw new \Exception('Codigo do pedido invalido');
            }

            $order = $this->getEntityManager()->find('Model\Entity\Order', $id);
            if(!$id){
                throw new \Exception('Pedido não encontrado');
            }

            $data = array();
            foreach($order->getOrderProducts() as $pro){

                $pw = $this->getEntityManager()->find('Model\Entity\ProductWarehouse', array('product' => $pro->getProduct(), 'warehouse' => $order->getWarehouse()));
                $arr['id'] = $pro->getProduct()->getId();
                $arr['name'] = $pro->getProduct()->getName();
                $arr['amount'] = $pro->getAmountRound();
                $arr['price'] = $pro->getPriceCurrency();
                $arr['subtotal'] = $pro->getSubtotalCurrency();
                $arr['discount'] = $pro->getDiscountCurrency();
                $arr['total'] = $pro->getTotalCurrency();
                $arr['stock'] = $pw->getAmountRound();
                $data[] = $arr;

            }

            $retorno = $data;

        }catch(\Exception $ex){
            $retorno['message'] = $ex->getMessage();
        }

        return new \Zend\View\Model\JsonModel($retorno);

    }

    public function confirmAction(){

        if(!$this->identity()->isAdmin()){
            return $this->redirectToDefaultRoute();
        }

        $id = $this->params()->fromRoute('id', false);
        if(!$id){
            throw new \Exception('Código do pedido invalido!');
        }

        $order = $this->getEntityManager()->find('Model\Entity\Order', $id);
        if(is_object($order)){
            throw new \Exception('Pedido não encontrado!');
        } else if($order->getStatus() !== \Model\Entity\Order::STATUS_WAITING){
            throw new \Exception('Pedido com status incorreto!');
        }

        foreach($order->getOrderProducts() as $pro){

            $pw = $this->getEntityManager()->find('Model\Entity\ProductWarehouse', array('product' => $pro, 'warehouse' => $order->getWarehouse()));
            $pw->subtract($pro->getAmount());
            $this->getEntityManager()->merge($pw);

        }

        $order->setStatus(\Model\Entity\Order::STATUS_OK);

        $this->getEntityManager()->merge($order);
        $this->getEntityManager()->flush();

        $this->flashMessenger()->addSuccessMessage("Pedido #" . $order->getId() . " confirmado com sucesso!");

        $this->redirect()->toRoute('application/default', array('controller' => 'order', 'action' => 'index'));

    }

    public function saveAction(){

        $response = array(
            'success' => false
        );

        try{

            //if(!$this->identity()->isAdmin()){
            //   throw new \Exception('Não autorizado');
            //}

            if(!$this->getRequest()->isPost()){
                throw new \Exception('Requisição inválida');
            }

            $config = $this->getConfig(); //Required

            $id = $this->params()->fromPost('id', false);

            if($id){

                $order = $this->getEntityManager()->find('Model\Entity\Order', $id);
                if(is_object($order) === false){
                    throw new \Exception('Pedido não encontrado!');
                } elseif ($order->getType() == \Model\Entity\Order::ATUALIZACAO) {
                    throw new \Exception('Não foi possível salvar suas informações!');
                }

            }else{

                $customer = $this->params()->fromPost('customer', false);
                if(!$customer){
                    throw new \Exception('Nenhum cliente selecionado');
                }

                $customerObj = $this->getEntityManager()->find('Model\Entity\Customer', $customer);
                if(!$customerObj){
                    throw new \Exception('Cliente não encontrado');
                }

                $warehouse = $this->getEntityManager()->getRepository('Model\Entity\Warehouse')->findOneByMain(true);

                $order = new \Model\Entity\Order();
                $order->setCustomer($customerObj);
                $order->setWarehouse($warehouse);
                $order->setUser($this->identity());
                $order->setOrderDate(new \Commons\Util\DateTimeBr());
                $order->setType(\Model\Entity\Order::PEDIDO);
                $order->setStatus(\Model\Entity\Order::STATUS_WAITING);
                $order->setCommission(0);

            }

            $order->setActive(true); //Required

            # Check Required Fields //Required
      			if (isset($config['salestool']['requiredParams']) && isset($config['salestool']['requiredParams']['order']) && is_array($config['salestool']['requiredParams']['order']) && count($config['salestool']['requiredParams']['order']) > 0) {
      				foreach ($config['salestool']['requiredParams']['order'] as $fieldToCheck) {
      					$paramToCheck = $this->params()->fromPost($fieldToCheck, false);
      					if(empty($paramToCheck)){
      						throw new \Exception('Um ou mais parâmetros obrigatórios não foram enviados!');
      					}
      				}
      			}

            # Set Customer
            $customer = $this->params()->fromPost('customer', false);
            if(!$customer){
                throw new \Exception('Nenhum cliente selecionado');
            }

            $customerObj = $this->getEntityManager()->find('Model\Entity\Customer', $customer);
            if(!$customerObj){
                throw new \Exception('Cliente não encontrado');
            }

            $order->setCustomer($customerObj);

            # Set Represented Company
            $representedCompany = $this->params()->fromPost('representedCompany', false);
            if($representedCompany){

                $representedCompanyObj = $this->getEntityManager()->find('Model\Entity\RepresentedCompany', $representedCompany);
                if(!$representedCompanyObj){
                    throw new \Exception('Representada não encontrado');
                }

                $order->setRepresentedCompany($representedCompanyObj);

            }

            # Set User
            if(!$order->getId()){
                $user = $this->params()->fromPost('user', false);
                if($user){
                    $userObj = $this->getEntityManager()->find('Model\Entity\User', $user);
                    if($userObj){
                        $order->setUser($userObj);
                    }
                }
            }

            # Set Payment method
            $paymentMethodId = $this->params()->fromPost('paymentMethod', false);
            if($paymentMethodId){

                $paymentMethod = $this->getEntityManager()->find('Model\Entity\PaymentMethod', $paymentMethodId);
                if($paymentMethod){
                    $order->setPaymentMethod($paymentMethod);
                }

            }

            # Set Payment condition
            $paymentConditionId = $this->params()->fromPost('paymentCondition', false);
            if($paymentMethodId){

                $paymentCondition = $this->getEntityManager()->find('Model\Entity\PaymentCondition', $paymentConditionId);
                if($paymentCondition){
                    $order->setPaymentCondition($paymentCondition);
                }

            }

            $products = $this->params()->fromPost('orderProducts',false);
            if(!$products || (is_array($products) && count($products) <= 0)){
                throw new \Exception("É necessário adicionar pelo menos um produto no pedido.");
            }

            if(!$order->getId()){
                $order->setSubtotal(0.0);
                $order->setDiscount(0.0);
                $order->setTotal(0.0);
                $this->getEntityManager()->persist($order);
                $this->getEntityManager()->flush();
            }

            // check products list from form first
            // if not add product
            foreach($products as $pro){

                $hasProduct = false;

                foreach($order->getOrderProducts() as $op){

                    if($op->getProduct()->getId() == $pro['id']){

                        $hasProduct = true;

                        if(!$pro['amount']){
                            $pro['amount'] = 0;
                        }

                        if($pro['amount'] > 0) {

                            $op->setObs($pro['obs']);
                            $op->setAmount($pro['amount']);
                            $op->setPrice($pro['price']);

                            $op->setSubtotal($op->getAmount() * $op->getPrice());

                            if ($pro['discount'] && $pro['discount'] > 0) {
                                $op->setDiscount($pro['discount']);
                                $op->setTotal($op->getSubtotal() - $op->getDiscount());
                            } else {
                                $op->setDiscount(0.0);
                                $op->setTotal($op->getSubtotal());
                            }

                            # Variants
                            //if($pro['variantsType']) {}
                            if (isset($pro['orderProductVariants']) && is_array($pro['orderProductVariants']) && count($pro['orderProductVariants']) > 0) {
                                foreach ($pro['orderProductVariants'] as $variant) {
                                $hasVariant = false;
                                foreach ($op->getOrderProductVariants() as $existingVariant) {
                                    if ($variant['id'] == $existingVariant->getId()) {
                                        $hasVariant = true;
                                        $existingVariant->setColor(($variant['color'] != null ? $this->getEntityManager()->find('Model\\Entity\\Color', $variant['color']) : null));
                                        $existingVariant->setSize(($variant['size'] != null ? $this->getEntityManager()->find('Model\\Entity\\Size', $variant['size']) : null));
                                        $existingVariant->setAmount($variant['amount']);
                                    }
                                }
                                if (!$hasVariant) {
                                    $orderProductVariant = new OrderProductVariant();
                                    $orderProductVariant->setColor(($variant['color'] != null ? $this->getEntityManager()->find('Model\\Entity\\Color', $variant['color']) : null));
                                    $orderProductVariant->setSize(($variant['size'] != null ? $this->getEntityManager()->find('Model\\Entity\\Size', $variant['size']) : null));
                                    $orderProductVariant->setAmount($variant['amount']);
                                    $orderProductVariant->setOrderProduct($op);
                                    //$this->getEntityManager()->persist($orderProductVariant);
                                    $op->getOrderProductVariants()->add($orderProductVariant);
                                }
                            }
                            }

                            $this->getEntityManager()->merge($op);

                        }else{
                            $this->getEntityManager()->remove($op);
                        }
                    }
                }

                if(!$hasProduct){

                    if($pro['amount'] > 0)
                    {
                        $product = $this->getEntityManager()->find('Model\Entity\Product', $pro['id']);
                        if(!$product){
                            throw new \Exception('Erro ao adicionar produto');
                        }

                        $op = new \Model\Entity\OrderProduct();
                        $op->setActive(true); //Required
                        $op->setProduct($product);
                        $op->setOrder($order);

                        if ($product->getCode() != '')
                            $op->setCodeProduct($product->getCode());

                        if(!$pro['amount']){
                            $pro['amount'] = 0;
                        }

                        $op->setWarehouse($order->getWarehouse());
                        $op->setAmount($pro['amount']);
                        $op->setPrice($pro['price']);
                        $op->setObs($pro['obs']);

                        $op->setSubtotal( $op->getAmount() * $op->getPrice() );

                        if($pro['discount'] && $pro['discount'] > 0){
                            $op->setDiscount($pro['discount']);
                            $op->setTotal( $op->getSubtotal() - $op->getDiscount() );
                        }else{
                            $op->setDiscount(0.0);
                            $op->setTotal( $op->getSubtotal() );
                        }

                        if (isset($pro['orderProductVariants']) && is_array($pro['orderProductVariants']) && count($pro['orderProductVariants']) > 0) {  //Required
              							foreach($pro['orderProductVariants'] as $variant)
              							{
              								$orderProductVariant = new OrderProductVariant();
              								$orderProductVariant->setColor( ($variant['color'] != null ? $this->getEntityManager()->find('Model\\Entity\\Color', $variant['color']) : null) );
              								$orderProductVariant->setSize( ($variant['size'] != null ? $this->getEntityManager()->find('Model\\Entity\\Size', $variant['size']) : null) );
              								$orderProductVariant->setAmount($variant['amount']);
              								$orderProductVariant->setOrderProduct($op);
              								$op->getOrderProductVariants()->add($orderProductVariant);
              							}
            						}

                        $this->getEntityManager()->persist($op);

                    }
                }

            }

            // Check for removed records
            foreach($order->getOrderProducts() as $op){

                $hasProduct = false;
                foreach($products as $pro){
                    if($op->getProduct()->getId() == $pro['id']){
                        $hasProduct = true;
                    }
                }

                if(!$hasProduct){
                    $this->getEntityManager()->remove($op);
                }

            }

            $this->getEntityManager()->merge($order);
            $this->getEntityManager()->flush();

            $this->getEntityManager()->refresh($order);

            $orderSum = 0;
            foreach($order->getOrderProducts() as $op){
                $orderSum += $op->getTotal();
            }

            $orderInfo = $this->params()->fromPost('info', "");
            $orderDiscount = $this->params()->fromPost('discount', 0);

            $order->setInfo($orderInfo);
            $order->setDiscount($orderDiscount);
            $order->setSubtotal($orderSum);
            $order->setTotal($orderSum - $orderDiscount);

            $this->getEntityManager()->merge($order);
            $this->getEntityManager()->flush();

            $action = $this->params()->fromPost('action', FALSE);

            if($action != FALSE && $action == 'confirm'){

                $userRepresentedCompany = null;
                if($order->getRepresentedCompany() && $order->getUser()){
                    $userRepresentedCompany = $this->getEntityManager()->getRepository('Model\\Entity\\UserRepresentedCompany')->findOneBy(array(
                        'representedCompany' => $order->getRepresentedCompany(),
                        'user' => $order->getUser()
                    ));
                }

                //Check for order discount percent
                $orderDiscountPercent = $order->getDiscountPercent();

                $totalCommission = 0;
                foreach($order->getOrderProducts() as $op){

                    $pw = $this->getEntityManager()->find('Model\Entity\ProductWarehouse', array('product' => $op->getProduct(), 'warehouse' => $order->getWarehouse()));
                    $pw->subtract($op->getAmount());
                    $this->getEntityManager()->merge($pw);

                    // Calculate commissions
                    $commissionValue = 0;
                    if($op->getProduct()->getCommission() && $op->getProduct()->getCommission() > 0){

                        $commissionValue = $op->getProduct()->getCommission();

                    }else if($userRepresentedCompany && $userRepresentedCompany->getCommission() > 0){

                        $commissionValue = $userRepresentedCompany->getCommission();

                    }

                    if($commissionValue && $commissionValue > 0){
                        $opTotal = $op->getTotal();
                        if($orderDiscountPercent > 0){
                            $opTotal = $opTotal - (($orderDiscountPercent/100) * $opTotal);
                        }
                        $c = ($commissionValue / 100) * $opTotal;
                        $op->setCommission($c);
                        $totalCommission += $c;
                        // $this->getEntityManager()->merge($op);
                    }

                }

                $order->setCommission($totalCommission);
                $order->setStatus(\Model\Entity\Order::STATUS_OK);

                $this->getEntityManager()->merge($order);
                $this->getEntityManager()->flush();

            }

            $response['success'] = true;
            $response['id'] = $order->getId();
            $response['status'] = $order->getStatus();
            $response['entity'] = $order;

            return $this->createCustomResponse($this->serialize($response));

        }catch(\Exception $ex){
            $response['success'] = false; //Required
            $response['message'] = $ex->getMessage(); //Required
        }

        //return new JsonModel($response); //Required
		    return $this->createCustomResponse($this->serialize($response)); //Required

    }

    public function addAction()
    {

        $entity = new $this->entityClass;
        $builder = new \DoctrineORMModule\Form\Annotation\AnnotationBuilder($this->getEntityManager());
        $form = $builder->createForm($entity);
        $form->setHydrator(new DoctrineHydrator($this->getEntityManager(), $this->getEntityClass()));

        $request = $this->getRequest();

        if ($request->isPost()){

            $form->bind($entity);
            $form->setData($request->getPost());

            if($form->isValid()){

                $hydrator = new DoctrineHydrator($this->getEntityManager(), $this->getEntityClass());
                $hydrator->hydrate($request->getPost()->toArray(), $entity);

                $this->getEntityManager()->persist($entity);
                $this->getEntityManager()->flush();

                $this->flashMessenger()->addSuccessMessage('Registro inserido com sucesso!');

                return $this->redirectToDefaultRoute();
            }


        }

        return new ViewModel(array('form' => $form));
    }

    public function editAction()
    {
        $id = null;
        $request = $this->getRequest();

        if ($request->isPost()){
            $id = $this->params()->fromPost('id', false);
        }else{
            $id = $this->params()->fromRoute('id', false);
        }

        $entity = $this->getEntityManager()->find($this->getEntityClass(), $id);

        if(is_object($entity) === false){
            throw new \Exception('Pedido não encontrado!');
        } elseif ($entity->getType() == \Model\Entity\Order::ATUALIZACAO) {
            throw new \Exception('Não é possível editar este pedido!');
        }

        $builder = new \DoctrineORMModule\Form\Annotation\AnnotationBuilder($this->getEntityManager());
        $form = $builder->createForm($entity);
        $form->setHydrator(new DoctrineHydrator($this->getEntityManager(), $this->getEntityClass()));

        if ($request->isPost()){

            $entity = $this->getEntityManager()->find($this->getEntityClass(), $this->params()->fromPost('id'));

            $hydrator = new DoctrineHydrator($this->getEntityManager(), $this->getEntityClass());
            $hydrator->hydrate($request->getPost()->toArray(), $entity);
            //$usuario->populate($form->getData());

            $this->getEntityManager()->merge($entity);
            $this->getEntityManager()->flush();

            $this->flashMessenger()->addSuccessMessage('Registro alterado com sucesso!');

            return $this->redirectToDefaultRoute();

        }else{
            $form->bind($entity);
        }

        return new ViewModel(array('form' => $form, 'entity' => $entity));
    }

    public function removeAction()
    {
        try{

            $id = $this->params()->fromRoute('id', false);
            if(!$id){
                throw new \Exception('Pedido não encontrado');
            }

            $entity = $this->getEntityManager()->find($this->getEntityClass(), $id);
            if(!$entity){
                throw new \Exception('Pedido não encontrado');
            }

            /*
            foreach($entity->getOrderProducts() as $ent){

                $pw = $this->getEntityManager()->find('Model\Entity\ProductWarehouse', array('product' => $ent->getProduct(), 'warehouse' => $ent->getWarehouse()));
                $pw->increase($ent->getAmount());
                $this->getEntityManager()->merge($pw);

            }
            */

            $this->getEntityManager()->remove($entity);
            $this->getEntityManager()->flush();
            $this->flashMessenger()->addSuccessMessage('Registro removido com sucesso!');
            return $this->redirectToDefaultRoute();

        } catch (\Doctrine\DBAL\DBALException $ex) {
            $this->flashMessenger()->addErrorMessage("Não foi possível remover o registro<br/>Pode ser que o registro que você está tentando remover possua relações com outros registros que impedem sua remoção.");
        } catch (\Exception $ex) {
            $this->flashMessenger()->addErrorMessage("Não foi possível remover o registro: {$ex->getMessage()}");
        }

        return $this->redirectToDefaultRoute();
    }

    public function cancelAction()
    {
        $id = $this->params()->fromRoute('id', false);

        if($id){

            $entity = $this->getEntityManager()->find($this->getEntityClass(), $id);
            if(is_object($entity)) {

                if ($entity->getStatus() == \Model\Entity\Order::STATUS_OK) {

                    foreach ($entity->getOrderProducts() as $ent) {

                        $pw = $this->getEntityManager()->find('Model\Entity\ProductWarehouse', array('product' => $ent->getProduct(), 'warehouse' => $ent->getWarehouse()));
						if (is_object($pw)) {
							$pw->increase($ent->getAmount());
							$this->getEntityManager()->merge($pw);
						}

                    }

                    $entity->setStatus(\Model\Entity\Order::STATUS_CANCELED);
                    $this->getEntityManager()->merge($entity);
                    $this->getEntityManager()->flush();
                    $this->flashMessenger()->addSuccessMessage('Pedido cancelado com sucesso!');
                    return $this->redirectToDefaultRoute();

                }else{
                    $this->flashMessenger()->addErrorMessage('O pedido selecionado não foi confirmado anteriormente.');
                }

            }else{
                $this->flashMessenger()->addErrorMessage('Erro ao remover pedidos. Código não encontrado.');
            }

        }else{
            $this->flashMessenger()->addErrorMessage('Erro ao remover pedidos. Código não encontrado.');
        }

        return $this->redirectToDefaultRoute();
    }

    public function getAppSyncInfoAction(){

        $retorno = array('error' => true, 'message' => '');

        try{

            $productId = $this->params()->fromPost('product', false);
            if(!$productId){
                throw new \Exception('Código do produto inválido');
            }

            $product = $this->getEntityManager()->find('Model\Entity\Product', $productId);
            if(!$product){
                throw new \Exception('Produto não encontrado');
            }

            $warehouseId = $this->params()->fromPost('warehouse', false);
            if(!$warehouseId){
                throw new \Exception('Código do estoque inválido');
            }

            $warehouse = $this->getEntityManager()->find('Model\Entity\Warehouse', $warehouseId);
            if(!$warehouse){
                throw new \Exception('Estoque não encontrado');
            }

            $pw = $this->getEntityManager()->find('Model\Entity\ProductWarehouse', array('product' => $product, 'warehouse' => $warehouse));
            if(!$pw){
                throw new \Exception('Estoque de produto não encontrado');
            }

            $arr = array();
            $arr['id'] = $product->getId();
            $arr['name'] = $product->getName();
            $arr['price'] = $product->getPriceCurrency();;
            $arr['stock'] = $pw->getAmountRound();

            return new \Zend\View\Model\JsonModel($arr);

        }catch(\Exception $ex){
            $retorno['message'] = $ex->getMessage();
        }

        return new \Zend\View\Model\JsonModel($retorno);

    }

    public function printAction(){

        try{

            $this->layout('layout/print');

            $id = $this->params()->fromRoute('id', false);
            if(!$id){
                throw new \Exception("Id inválido");
            }

            $entity = $this->getEntityManager()->find('Model\Entity\Order', $id);
            if(!$entity){
                throw new \Exception("Nenhuma pedido encontrada com este id");
            }

            $company = $this->getEntityManager()->getRepository('Model\Entity\Company')->getDefault();

            # Select products ordered by name
            $products = $this->getEntityManager()->createQueryBuilder()
                ->select("obj")
                ->from("Model\\Entity\\OrderProduct","obj")
                ->join("obj.product","prd")
                ->where("obj.order = :ord")
                ->orderBy("prd.name")
                ->setParameter("ord", $entity)
                ->getQuery()->getResult();

            $variants = array();
            if (count($products) > 0) {
                foreach ($products as $product) {
                    $variants[$product->getId()] = $product->getOrderProductVariants();
                }
            }

            return new ViewModel(array('entity' => $entity, 'company' => $company, 'products' => $products, 'variants' => $variants));

        }catch(\Exception $ex){

            $this->flashMessenger()->addErrorMessage("Ocorreu um erro ao processar sua solicitação. Descrição: " . $ex->getMessage());

        }

        return $this->redirectToDefaultRoute();

    }

}
