<?php
/**
 * 
 * Abstract application controller
 * 
 * @author Renato Peterman <renato.pet (at) gmail.com>
 *
 */

namespace Application\Controller;

use Zend\Mvc\Controller\AbstractActionController;
use Zend\View\Model\JsonModel;
use Zend\Http\Response\Stream;
use Zend\Http\Headers;

abstract class AbstractApplicationController extends AbstractActionController
{
    protected $em;
    protected $authManager;
    protected $configManager;
    protected $whiteList = array('login','download');

    /**
     * 
     * Verify if user was authenticated
     * 
     * @override
     */
    public function onDispatch(\Zend\Mvc\MvcEvent $e) {
        
        if(!$this->getAuthManager()->getIdentity() && !in_array($e->getRouteMatch()->getMatchedRouteName(), $this->whiteList)){
            return $this->redirect()->toRoute('login');
        }

        if($e->getRouteMatch()->getParam('controller') != 'Application\Controller\Agreement' && $e->getRouteMatch()->getParam('controller') != 'Application\Controller\Login')
        {
            $config = $this->getEntityManager()->getRepository('Model\\Entity\\Config')->findOneBy(array());
            if(!$config->getAdminHasAgreedTerms())
            {
                if($this->identity()->isAdmin())
                {
                    return $this->redirect()->toRoute("application/default", array("controller" => 'agreement', "action" => 'index'));
                }
                else
                {
                    return $this->redirect()->toRoute("application/default", array("controller" => 'agreement', "action" => 'warning'));
                }
            }
        }
        
        parent::onDispatch($e);
    }
    
    /**
     * 
     * Return a Doctrine Entity Manager instance
     * 
     * @return Doctrine\ORM\EntityManager
     */
    public function getEntityManager()
    {
        if (null === $this->em) {
            $this->em = $this->getServiceLocator()->get('Doctrine\ORM\EntityManager');
        }
        return $this->em;
    }
    
    /**
     * 
     * Return a Zend Authentication Service instance
     * 
     * @return Zend\Authentication\AuthenticationService
     */
    public function getAuthManager()
    {
        if (null === $this->authManager) {
            $this->authManager = $this->getServiceLocator()->get('Zend\Authentication\AuthenticationService');
        }
        return $this->authManager;
    }

    /**
     *
     * Return Config service
     *
     * @return ConfigService
     */
    public function getConfig()
    {
        if (null === $this->configManager) {
            $this->configManager = $this->getServiceLocator()->get('Config');
        }
        return $this->configManager;
    }

    /**
     *
     * Return user limit
     *
     * @return ConfigService
     */
    public function getUserLimit()
    {
        $config = $this->getConfig();
        if(!array_key_exists('salestool', $config)){
            return false;
        }
        return $config['salestool']['userLimit'];
    }

    /**
     *
     * Return device limit
     *
     * @return ConfigService
     */
    public function getDeviceLimit()
    {
        $config = $this->getConfig();
        if(!array_key_exists('salestool', $config)){
            return false;
        }
        return $config['salestool']['deviceLimit'];
    }

    /**
     * Check if user limit was reached
     * @return bool
     */
    public function isReachedUserLimit(){
        if(!$this->getUserLimit()){
            return false;
        }

        $usersCount = $this->getEntityManager()->createQuery("SELECT COUNT(obj.id) FROM Model\\Entity\\User obj")->getSingleScalarResult();
        if($usersCount >= $this->getUserLimit()){
            return true;
        }

        return false;
    }

    /**
     * Check if device limit was reached
     * @return bool
     */
    public function isReachedDeviceLimit(){
        if(!$this->getDeviceLimit()){
            return false;
        }

        $devicesCount = $this->getEntityManager()->createQuery("SELECT COUNT(obj.id) FROM Model\\Entity\\Device obj")->getSingleScalarResult();
        if($devicesCount >= $this->getDeviceLimit()){
            return true;
        }

        return false;
    }
    
    /**
     * 
     * Create a stream response
     * 
     * @param string $file
     * @param string $contentType
     * @return \Zend\Http\Response\Stream
     */
    public function createFileResponse($file, $contentType){
        $response = new Stream();
        $response->setStream(fopen($file, 'r'));
        $response->setStatusCode(200);
        $headers = new Headers();
        $headers->addHeaderLine('Content-Transfer-Encoding', 'binary')
                ->addHeaderLine('Content-Type', $contentType)
                ->addHeaderLine('Content-Length', filesize($file));

        $response->setHeaders($headers);
        return $response;
    }
    
    /**
     * 
     * Create a JSON Response
     * 
     * @param array $array
     */
    public function createJsonResponse($array){
        return new JsonModel($array);
    }
    
    public function __get($name) {
        if($name == "entityClass"){
            return $this->getEntityClass();
        }
    }
    
    public function getEntityClass(){
        $parts = explode("\\", get_called_class());
        $name = str_replace("Controller", "", $parts[count($parts)-1]);
        return "Model\\Entity\\" . $name; 
    }

    /**
     * Get default entity config
     * @private
     * @return mixed
     */
    public function getDefaultConfig(){
        return $this->getEntityManager()->getRepository('Model\\Entity\\Config')->findOneBy(array());
    }
    
    public function redirectToDefaultRoute($action = null){
        if(!$action){
            $action = "index";
        }
        $parts = explode("\\", get_called_class());
        $name = str_replace("Controller", "", $parts[count($parts)-1]);
        return $this->redirect()->toRoute("application/default", array("controller" => strtolower($name), "action" => $action));
    }

    /**
     * Create a custom controller response
     * @param $content
     * @return mixed
     */
    protected function createCustomResponse($content){

        $format = $this->getEvent()->getRouteMatch()->getParam('format');
        $response = $this->getResponseWithHeader();
        if(!$format){
            $response->getHeaders()->addHeaderLine('Content-Type','application/json');
        }else{
            if($format == 'json'){
                $response->getHeaders()->addHeaderLine('Content-Type','application/json');
            }else if($format == 'xml'){
                $response->getHeaders()->addHeaderLine('Content-Type','application/xml');
            }
        }

        return $response->setContent($content);
    }

    /**
     * Get response object with headers
     * @return \Zend\Stdlib\ResponseInterface
     */
    protected function getResponseWithHeader()
    {
        $response = $this->getResponse();
        $response->getHeaders()
            ->addHeaderLine('Access-Control-Allow-Origin','*')
            ->addHeaderLine('Access-Control-Allow-Methods','POST PUT DELETE GET');
        return $response;
    }

    /**
     * Serialize entity using jms serializer
     * @param $entity
     * @param string $format
     */
    protected function serialize($entity, $format = 'json'){
        return $this->getServiceLocator()->get('jms_serializer.serializer')->serialize($entity, $format);
    }
    
}
