<?php

namespace Model\Entity;

use Doctrine\ORM\Mapping as ORM;
use Doctrine\Common\Collections\ArrayCollection;
use JMS\Serializer\Annotation as JMS;
use Zend\Form\Annotation;
use ZF\OAuth2\Doctrine\Entity\UserInterface;

/**
 * @ORM\Entity
 * @ORM\Table(name="users")
 * @Annotation\Name("user")
 * @Annotation\Attributes({"class":"form-horizontal form-validate"})
 */
class User extends AbstractEntity implements UserInterface {

    const PATH_PROFILE_PICTURE = './data/profile';
    const PATH_NO_PICTURE = './data/profile/no-picture.jpg';

    /**
     * @ORM\Id
     * @ORM\Column(name="id", type="bigint")
     * @ORM\GeneratedValue(strategy="AUTO")
     * @Annotation\Exclude()
     */
    protected $id;

    /**
     * @ORM\Column(name="code", type="string", length=40, nullable=true)
     */
    protected $code;

    /**
     * @ORM\Column(name="name", type="string")
     * @Annotation\Type("Zend\Form\Element\Text")
     * @Annotation\Required({"required":"true" })
     * @Annotation\Attributes({"type":"text", "class":"form-control", "placeholder":"Obrigatório"})
     * @Annotation\Options({"label":"Nome:"})
     */
    protected $name;

    /**
     * @ORM\Column(name="email", type="string", unique=true, nullable=true)
     * @Annotation\Type("Zend\Form\Element\Text")
     * @Annotation\Attributes({"type":"text", "class":"form-control", "placeholder":""})
     * @Annotation\Options({"label":"E-mail:"})
     */
    protected $email;

    /**
     * @ORM\Column(name="username", type="string", unique=true)
     * @Annotation\Type("Zend\Form\Element\Text")
     * @Annotation\Required({"required":"true" })
     * @Annotation\Attributes({"type":"text", "class":"form-control", "placeholder":"Obrigatório"})
     * @Annotation\Options({"label":"Nome de Usuário:"})
     */
    protected $username;

    /**
     * @ORM\Column(name="password", type="string")
     * @Annotation\Type("Zend\Form\Element\Password")
     * @Annotation\Required({"required":"true" })
     * @Annotation\Attributes({"type":"password", "class":"form-control", "placeholder":"Obrigatório"})
     * @Annotation\Options({"label":"Senha:"})
     *
     * @Annotation\Exclude()
     */
    protected $password;

    /**
     * @ORM\Column(name="admin", type="boolean", nullable=true)
     * @Annotation\AllowEmpty(true)
     * @Annotation\Type("Zend\Form\Element\Checkbox")
     * @Annotation\Options({"label":"Admin?"})
     */
    protected $admin;

    /**
     * @ORM\Column(name="track_location", type="boolean", nullable=true, options={"default": false})
     * @Annotation\AllowEmpty(true)
     * @Annotation\Type("Zend\Form\Element\Checkbox")
     * @Annotation\Options({"label":"Rastrear usuário"})
     */
    protected $trackLocation;

    /**
     * @ORM\Column(name="token_password", type="string", nullable=true)
     * @Annotation\Exclude()
     */
    protected $tokenPassword;

    /**
     * @ORM\Column(name="seller_id", type="string", nullable=true)
     * @Annotation\Type("Zend\Form\Element\Text")
     * @Annotation\Attributes({"type":"text", "class":"form-control", "placeholder":""})
     * @Annotation\Options({"label":"Código Vendedor:"})
     */
    protected $sellerId;

    /**
     * @ORM\Column(name="discount_maximum", type="decimal", precision=19, scale=3, nullable=true)
     * @Annotation\Type("Zend\Form\Element\Text")
     * @Annotation\Required({"required":"true" })
     * @Annotation\Attributes({"type":"text", "class":"form-control mask-money", "placeholder":"Obrigatório"})
     * @Annotation\Options({"label":"Desconto Máximo:"})
     */
    protected $discountMaximum;

    /**
     * @ORM\Column(name="image", type="string", nullable=true)
     * @Annotation\Exclude()
     */
    protected $image;

    /**
     * @ORM\OneToMany(targetEntity="Model\Entity\OAuth2\Client", mappedBy="user", cascade={"persist", "merge", "remove"})
     * @ORM\JoinColumn(name="id", referencedColumnName="id_user")
     * @Annotation\Exclude()
     *
     * @JMS\Exclude()
     */
    protected $client;

    /**
     * @ORM\OneToMany(targetEntity="Model\Entity\OAuth2\AccessToken", mappedBy="user", cascade={"persist", "merge", "remove"})
     * @ORM\JoinColumn(name="id", referencedColumnName="id_user")
     * @Annotation\Exclude()
     *
     * @JMS\Exclude()
     */
    protected $accessToken;

    /**
     * @ORM\Column(name="authorization_code", type="string", nullable=true)
     * @Annotation\Exclude()
     *
     * @JMS\Exclude()
     */
    protected $authorizationCode;

    /**
     * @ORM\Column(name="refresh_token", type="string", nullable=true)
     * @Annotation\Exclude()
     *
     * @JMS\Exclude()
     */
    protected $refreshToken;

    /**
     * @ORM\OneToMany(targetEntity="Model\Entity\UserLocation", mappedBy="user", cascade={"persist", "merge", "remove"})
     * @ORM\JoinColumn(name="id", referencedColumnName="id_user")
     * @Annotation\Exclude()
     *
     * @JMS\Exclude()
     */
    protected $locations;

    /**
     * @ORM\OneToMany(targetEntity="Model\Entity\UserRepresentedCompany", mappedBy="user", cascade={"all"})
     * @ORM\JoinColumn(name="id", referencedColumnName="id_user")
     * @Annotation\Exclude()
     *
     * @JMS\Exclude()
     */
    protected $representedCompanies;

    /**
     * @ORM\OneToMany(targetEntity="Model\Entity\UserPriceList", mappedBy="user", cascade={"all"})
     * @ORM\JoinColumn(name="id", referencedColumnName="id_user")
     * @Annotation\Exclude()
     *
     * @JMS\Exclude()
     */
    protected $userPriceLists;

    /**
     * @ORM\OneToMany(targetEntity="Model\Entity\UserCustomer", mappedBy="user", cascade={"all"})
     * @ORM\JoinColumn(name="id", referencedColumnName="id_user")
     * @Annotation\Exclude()
     *
     * @JMS\Exclude()
     */
    protected $userCustomers;

    public function __construct()
    {
        parent::__construct();
        $this->active = true;
        $this->admin = false;
        $this->setTrackLocation(false);
        $this->userPriceLists = new ArrayCollection();
        $this->userCustomers = new ArrayCollection();
    }

    public function getSellerId() {
        return $this->sellerId;
    }

    public function setSellerId($sellerId) {
        $this->sellerId = $sellerId;
    }

    public function getDiscountMaximum() {
        return $this->discountMaximum;
    }

    public function setDiscountMaximum($discountMaximum) {
        $this->discountMaximum = $discountMaximum;
    }

    public function getId() {
        return $this->id;
    }

    public function setId($id) {
        $this->id = $id;
    }

    /**
     * Set the value of code.
     *
     * @param string $code
     * @return \Model\Entity\Customer
     */
    public function setCode($code)
    {
        $this->code = $code;

        return $this;
    }

    /**
     * Get the value of code.
     *
     * @return string
     */
    public function getCode()
    {
        return $this->code;
    }

    public function getName() {
        return $this->name;
    }

    public function setName($name) {
        $this->name = $name;
    }

    public function getEmail() {
        return $this->email;
    }

    public function setEmail($email) {
        $this->email = $email;
    }

    public function getUsername() {
        return $this->username;
    }

    public function setUsername($username) {
        $this->username = $username;
    }

    public function getPassword() {
        return $this->password;
    }

    public function setPassword($password)
    {
        $bcrypt = new \Zend\Crypt\Password\Bcrypt();
        $this->password = $bcrypt->create( hash('sha256', $password) );
        return $this;
    }

    public function setPasswordHashed($plaintextPassword)
    {
        $bcrypt = new \Zend\Crypt\Password\Bcrypt();
        $this->password = $bcrypt->create($plaintextPassword);
        return $this;
    }

    public function getTokenPassword() {
        return $this->tokenPassword;
    }

    public function setTokenPassword($tokenPassword) {
        $this->tokenPassword = $tokenPassword;
    }

    public function isAdmin() {
        return $this->admin;
    }

    public function getAdmin() {
        return $this->admin;
    }

    public function setAdmin($admin) {
        $this->admin = $admin;
    }

    public function getImage()
    {
        return $this->image;
    }

    public function setImage($image)
    {
        $this->image = $image;
    }

    public function setImageFile($image)
    {
        $this->image = $image;
    }

    public function getProfilePicturePath(){
        return self::getPathForFile($this->getImage());
    }

    public static function getPathForFile($filename = ''){
        return self::PATH_PROFILE_PICTURE . "/" . $filename;
    }

    /**
     * @return mixed
     */
    public function getRefreshToken()
    {
        return $this->refreshToken;
    }

    /**
     * @param mixed $refreshToken
     */
    public function setRefreshToken($refreshToken)
    {
        $this->refreshToken = $refreshToken;
    }

    /**
     * @return mixed
     */
    public function getAuthorizationCode()
    {
        return $this->authorizationCode;
    }

    /**
     * @param mixed $authorizationCode
     */
    public function setAuthorizationCode($authorizationCode)
    {
        $this->authorizationCode = $authorizationCode;
    }

    /**
     * @return mixed
     */
    public function getAccessToken()
    {
        return $this->accessToken;
    }

    /**
     * @param mixed $accessToken
     */
    public function setAccessToken($accessToken)
    {
        $this->accessToken = $accessToken;
    }

    /**
     * @return mixed
     */
    public function getClient()
    {
        return $this->client;
    }

    /**
     * @param mixed $client
     */
    public function setClient($client)
    {
        $this->client = $client;
    }

    /**
     * @return mixed
     */
    public function getLocations()
    {
        return $this->locations;
    }

    /**
     * @param mixed $locations
     */
    public function setLocations($locations)
    {
        $this->locations = $locations;
    }

    /**
     * @return mixed
     */
    public function getTrackLocation()
    {
        return $this->trackLocation;
    }

    /**
     * @param mixed $trackLocation
     */
    public function setTrackLocation($trackLocation)
    {
        $this->trackLocation = $trackLocation;
    }

    /**
     * @return mixed
     */
    public function getRepresentedCompanies()
    {
        return $this->representedCompanies;
    }

    /**
     * @param mixed $representedCompanies
     */
    public function setRepresentedCompanies($representedCompanies)
    {
        $this->representedCompanies = $representedCompanies;
    }

    /**
     * @param $list
     */
    public function addRepresentedCompanies($list){
        foreach($list as $obj){
            $this->representedCompanies->add($obj);
        }
    }

    /**
     * @param $list
     */
    public function removeRepresentedCompanies($list){
        foreach($list as $obj){
            $this->representedCompanies->removeElement($obj);
        }
    }

    /**
     * @param $obj
     * @return bool
     */
    public function hasRepresentedCompanies($obj){

        if($obj){
            foreach($this->getRepresentedCompanies() as $entity){
                if($entity->getId() == $obj->getId()){
                    return true;
                }
            }
        }

        return false;
    }


    public function addUserPriceList(UserPriceList $userPriceList)
    {
        $this->userPriceLists[] = $userPriceList;

        return $this;
    }

    public function removeUserPriceList(UserPriceList $userPriceList)
    {
        $this->userPriceLists->removeElement($userPriceList);

        return $this;
    }

    public function getUserPriceLists()
    {
        return $this->userPriceLists;
    }


    public function addUserCustomer(UserCustomer $customer)
    {
        $this->userCustomers[] = $customer;

        return $this;
    }

    public function removeUserCustomer(UserCustomer $customer)
    {
        $this->userCustomers->removeElement($customer);

        return $this;
    }

    public function getUserCustomers()
    {
        return $this->userCustomers;
    }


    /**
     * Verify if password match
     * @param Model\Entity\User $user
     * @param string            $password
     * @param boolean           $plain
     * @return type
     */
    public static function hashPassword($user, $password, $plain = true)
    {
        if($plain){
            $bcrypt = new \Zend\Crypt\Password\Bcrypt();
            return ($bcrypt->verify(hash('sha256', $password), $user->getPassword()));
        }else{
            $bcrypt = new \Zend\Crypt\Password\Bcrypt();
            return ($bcrypt->verify($password, $user->getPassword()));
        }
    }

    /**
     * Get simple array copy from object
     * @return array
     */
    public function getArrayCopy() {
        $arr = parent::getArrayCopy();
        //unset($arr['password']);
        unset($arr['tokenPassword']);
        unset($arr['created']);
        unset($arr['updated']);
        return $arr;
    }

    /**
     * __toString()
     * @return string
     */
    public function __toString() {
        return $this->getId() . ' - ' . $this->getName();
    }

}

?>
