<?php

namespace Model\Entity\OAuth2;

use Doctrine\ORM\Mapping as ORM;
use Zend\Math\Rand;
use Zend\Stdlib\ArraySerializableInterface;
use Doctrine\Common\Collections\ArrayCollection;
use JMS\Serializer\Annotation as JMS;

/**
 * Client
 *
 * @ORM\Entity()
 * @ORM\Table(name="oauth2_client")
 */
class Client implements ArraySerializableInterface
{
    /**
     * @ORM\Column(name="client_id", type="string", length=255, nullable=true)
     *
     * @var string
     */
    private $clientId;

    /**
     * @ORM\Column(name="secret", type="string", length=255, nullable=true)
     *
     * @JMS\Exclude()
     */
    private $secret;

    /**
     * @ORM\Column(name="secret_plain", type="string", length=255, nullable=true)
     *
     * @var string
     */
    private $secretPlain;

    /**
     * @ORM\Column(name="redirect_uri", type="string", length=255, nullable=true)
     *
     * @var string
     */
    private $redirectUri;

    /**
     * @ORM\Column(name="grant_type", type="array", nullable=true)
     *
     * @var array
     */
    private $grantType;

    /**
     * @ORM\Column(name="client_scope", type="string", nullable=true)
     *
     * @var string
     */
    private $clientScope;

    /**
     * @ORM\Id
     * @ORM\Column(type="integer")
     * @ORM\GeneratedValue(strategy="AUTO")
     *
     * @var integer
     */
    private $id;

    /**
     * @ORM\OneToMany(targetEntity="Model\Entity\OAuth2\AccessToken", mappedBy="client")
     * @ORM\JoinColumn(name="id", referencedColumnName="client_id")
     *
     * @var \Doctrine\Common\Collections\Collection
     *
     * @JMS\Exclude()
     */
    private $accessToken;

    /**
     * @ORM\OneToMany(targetEntity="Model\Entity\OAuth2\RefreshToken", mappedBy="client")
     * @ORM\JoinColumn(name="id", referencedColumnName="client_id")
     *
     * @var \Doctrine\Common\Collections\Collection
     *
     * @JMS\Exclude()
     */
    private $refreshToken;

    /**
     * @ORM\OneToMany(targetEntity="Model\Entity\OAuth2\AuthorizationCode", mappedBy="client")
     * @ORM\JoinColumn(name="id", referencedColumnName="client_id")
     *
     * @var \Doctrine\Common\Collections\Collection
     *
     * @JMS\Exclude()
     */
    private $authorizationCode;

    /**
     * @ORM\OneToMany(targetEntity="Model\Entity\OAuth2\Jwt", mappedBy="client")
     * @ORM\JoinColumn(name="id", referencedColumnName="client_id")
     *
     * @var \Doctrine\Common\Collections\Collection
     *
     * @JMS\Exclude()
     */
    private $jwt;

    /**
     * @ORM\OneToMany(targetEntity="Model\Entity\OAuth2\Jti", mappedBy="client")
     * @ORM\JoinColumn(name="id", referencedColumnName="client_id")
     *
     * @var \Doctrine\Common\Collections\Collection
     *
     * @JMS\Exclude()
     */
    private $jti;

    /**
     * @ORM\OneToMany(targetEntity="Model\Entity\OAuth2\PublicKey", mappedBy="client")
     * @ORM\JoinColumn(name="id", referencedColumnName="client_id")
     *
     * @var \Doctrine\Common\Collections\Collection
     *
     * @JMS\Exclude()
     */
    private $publicKey;

    /**
     * @ORM\ManyToMany(targetEntity="Model\Entity\OAuth2\Scope", cascade={"remove"})
     * @ORM\JoinTable(name="oauth2_client_to_scope",
     *      joinColumns={@ORM\JoinColumn(name="client_id", referencedColumnName="id")},
     *      inverseJoinColumns={@ORM\JoinColumn(name="scope_id", referencedColumnName="id")}
     *      )
     *
     * @var \Doctrine\Common\Collections\Collection
     */
    private $scope;

    /**
     * @ORM\ManyToOne(targetEntity="Model\Entity\User", inversedBy="accessToken")
     * @ORM\JoinColumn(name="user_id", referencedColumnName="id", nullable=true)
     */
    private $user;

    /**
     * @ORM\Column(name="description", type="string", nullable=true)
     *
     * @var string
     */
    private $description;

    /**
     * @ORM\Column(name="active", type="boolean", nullable=true)
     */
    protected $active;

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->accessToken = new ArrayCollection();
        $this->refreshToken = new ArrayCollection();
        $this->authorizationCode = new ArrayCollection();
        $this->jwt = new ArrayCollection();
        $this->jti = new ArrayCollection();
        $this->publicKey = new ArrayCollection();
        $this->scope = new ArrayCollection();

        // Create ID and Secret
        $clientID = (rand()*10000) . mt_rand(1,9);
        $this->setClientId($clientID);

        // $this->clientId = Rand::getString(32, "0123456789");
        $secret = static::generateSecret();
        $this->setSecret($secret);
        $this->setSecretPlain($secret);
    }

    public function getArrayCopy()
    {
        return array(
            'id' => $this->getId(),
            'clientId' => $this->getClientId(),
            'secret' => $this->getSecret(),
            'redirectUri' => $this->getRedirectUri(),
            'grantType' => $this->getGrantType(),
            'scope' => $this->getScope(),
            'user' => $this->getUser(),
        );
    }

    public function exchangeArray(array $array)
    {
        foreach ($array as $key => $value) {
            switch ($key) {
                case 'clientId':
                    $this->setClientId($value);
                    break;
                case 'secret':
                    $this->setSecret($value);
                    break;
                case 'redirectUri':
                    $this->setRedirectUri($value);
                    break;
                case 'grantType':
                    $this->setGrantType($value);
                    break;
                case 'user':
                    $this->setUser($value);
                    break;
                case 'scope':
                    // Clear old collection
                    foreach ($value as $remove) {
                        $this->removeScope($remove);
                        $remove->removeClient($this);
                    }

                    // Add new collection
                    foreach ($value as $scope) {
                        $this->addScope($scope);
                        $scope->removeClient($this);
                    }
                    break;
                default:
                    break;
            }
        }

        return $this;
    }

    /**
     * Set clientId
     *
     * @param string $clientId
     * @return Client
     */
    public function setClientId($clientId)
    {
        $this->clientId = $clientId;

        return $this;
    }

    /**
     * Get clientId
     *
     * @return string
     */
    public function getClientId()
    {
        return $this->clientId;
    }

    /**
     * Set secret
     *
     * @param string $secret
     * @return Client
     */
    public function setSecret($secret)
    {
        $bcrypt = new \Zend\Crypt\Password\Bcrypt();
        $this->secret = $bcrypt->create($secret);
        return $this;
    }

    /**
     * Get secret
     *
     * @return string
     */
    public function getSecret()
    {
        return $this->secret;
    }

    /**
     * Set redirectUri
     *
     * @param string $redirectUri
     * @return Client
     */
    public function setRedirectUri($redirectUri)
    {
        $this->redirectUri = $redirectUri;

        return $this;
    }

    /**
     * Get redirectUri
     *
     * @return string
     */
    public function getRedirectUri()
    {
        return $this->redirectUri;
    }

    /**
     * Set grantType
     *
     * @param array $grantType
     * @return Client
     */
    public function setGrantType($grantType)
    {
        $this->grantType = $grantType;

        return $this;
    }

    /**
     * Get grantType
     *
     * @return array
     */
    public function getGrantType()
    {
        return $this->grantType;
    }

    /**
     * Set clientScope
     *
     * @param string $clientScope
     * @return Client
     */
    public function setClientScope($clientScope)
    {
        $this->clientScope = $clientScope;

        return $this;
    }

    /**
     * Get clientScope
     *
     * @return string
     */
    public function getClientScope()
    {
        return $this->clientScope;
    }

    /**
     * Get id
     *
     * @return integer
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Add accessToken
     *
     * @param AccessToken $accessToken
     * @return Client
     */
    public function addAccessToken(AccessToken $accessToken)
    {
        $this->accessToken[] = $accessToken;

        return $this;
    }

    /**
     * Remove accessToken
     *
     * @param AccessToken $accessToken
     */
    public function removeAccessToken(AccessToken $accessToken)
    {
        $this->accessToken->removeElement($accessToken);
    }

    /**
     * Get accessToken
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getAccessToken()
    {
        return $this->accessToken;
    }

    /**
     * Add refreshToken
     *
     * @param RefreshToken $refreshToken
     * @return Client
     */
    public function addRefreshToken(RefreshToken $refreshToken)
    {
        $this->refreshToken[] = $refreshToken;

        return $this;
    }

    /**
     * Remove refreshToken
     *
     * @param RefreshToken $refreshToken
     */
    public function removeRefreshToken(RefreshToken $refreshToken)
    {
        $this->refreshToken->removeElement($refreshToken);
    }

    /**
     * Get refreshToken
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getRefreshToken()
    {
        return $this->refreshToken;
    }

    /**
     * Add authorizationCode
     *
     * @param AuthorizationCode $authorizationCode
     * @return Client
     */
    public function addAuthorizationCode(AuthorizationCode $authorizationCode)
    {
        $this->authorizationCode[] = $authorizationCode;

        return $this;
    }

    /**
     * Remove authorizationCode
     *
     * @param AuthorizationCode $authorizationCode
     */
    public function removeAuthorizationCode(AuthorizationCode $authorizationCode)
    {
        $this->authorizationCode->removeElement($authorizationCode);
    }

    /**
     * Get authorizationCode
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getAuthorizationCode()
    {
        return $this->authorizationCode;
    }

    /**
     * Add jwt
     *
     * @param Jwt $jwt
     * @return Client
     */
    public function addJwt(Jwt $jwt)
    {
        $this->jwt[] = $jwt;

        return $this;
    }

    /**
     * Remove jwt
     *
     * @param Jwt $jwt
     */
    public function removeJwt(Jwt $jwt)
    {
        $this->jwt->removeElement($jwt);
    }

    /**
     * Get jwt
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getJwt()
    {
        return $this->jwt;
    }

    /**
     * Add jti
     *
     * @param Jti $jti
     * @return Client
     */
    public function addJti(Jti $jti)
    {
        $this->jti[] = $jti;

        return $this;
    }

    /**
     * Remove jti
     *
     * @param Jti $jti
     */
    public function removeJti(Jti $jti)
    {
        $this->jti->removeElement($jti);
    }

    /**
     * Get jti
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getJti()
    {
        return $this->jti;
    }

    /**
     * Add publicKey
     *
     * @param PublicKey $publicKey
     * @return Client
     */
    public function addPublicKey(PublicKey $publicKey)
    {
        $this->publicKey[] = $publicKey;

        return $this;
    }

    /**
     * Remove publicKey
     *
     * @param PublicKey $publicKey
     */
    public function removePublicKey(PublicKey $publicKey)
    {
        $this->publicKey->removeElement($publicKey);
    }

    /**
     * Get publicKey
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getPublicKey()
    {
        return $this->publicKey;
    }

    /**
     * Add scope
     *
     * @param Scope $scope
     * @return Client
     */
    public function addScope(Scope $scope)
    {
        $this->scope[] = $scope;

        return $this;
    }

    /**
     * Remove scope
     *
     * @param Scope $scope
     */
    public function removeScope(Scope $scope)
    {
        $this->scope->removeElement($scope);
    }

    /**
     * Get scope
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getScope()
    {
        return $this->scope;
    }

    /**
     * Set user
     *
     * @param $user
     * @return Client
     */
    public function setUser($user = null)
    {
        $this->user = $user;

        return $this;
    }

    /**
     * Get user
     *
     * @return User
     */
    public function getUser()
    {
        return $this->user;
    }

    /**
     * @return string
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * @param string $description
     */
    public function setDescription($description)
    {
        $this->description = $description;
    }

    /**
     * @return mixed
     */
    public function getActive()
    {
        return $this->active;
    }

    /**
     * @param mixed $active
     */
    public function setActive($active)
    {
        $this->active = $active;
    }

    /**
     * @return string
     */
    public function getSecretPlain()
    {
        return $this->secretPlain;
    }

    /**
     * @param string $secretPlain
     */
    public function setSecretPlain($secretPlain)
    {
        $this->secretPlain = $secretPlain;
    }

    /**
     * Update application secret
     */
    public function updateSecret(){
        $secret = static::generateSecret();
        $this->setSecret($secret);
        $this->setSecretPlain($secret);
    }

    /**
     * Generate secret
     * @return string
     */
    public static function generateSecret(){
        return sha1(uniqid(rand(), true));
    }
}
