<?php
/**
 * Index Controller
 *
 * @author Renato Peterman <renato.peterman@gestaoconex.com.br>
 * 
 */

namespace Application\Controller;

use Commons\Util\DateTimeBr;
use Commons\Util\StringUtils;
use Zend\View\Model\ViewModel;
use Zend\View\Model\JsonModel;

class IndexController extends AbstractApplicationController
{
    /**
     * Index action
     * @return ViewModel
     */
    public function indexAction()
    {
        $view = new ViewModel();
        return $view;
    }

    /**
     * Dashboard action
     * @return ViewModel
     */
    public function dashboardAction()
    {
        $view = new ViewModel();
        $view->setTerminal(true);
        return $view;
    }

    /**
     * Last orders list
     * @return JsonModel
     */
    public function lastOrdersAction(){
        
        $qb = $this->getEntityManager()->createQueryBuilder()
                ->select('obj')
                ->from('Model\Entity\Order', 'obj')
                ->orderBy('obj.id', 'desc')
                ->where('obj.status = :statusWaiting')
                ->setParameter('statusWaiting', \Model\Entity\Order::STATUS_WAITING)
                ->setMaxResults(6);
        
        if(!$this->identity()->isAdmin()){
            $qb->andWhere('obj.user = :user')->setParameter('user', $this->identity());
        }
        
        $orders = $qb->getQuery()->getResult();
        $response = array();
        foreach($orders as $obj){
            $response[] = array(
                'id' => $obj->getId(),
                'orderDate' => $obj->getOrderDateFormatted(),
                'user' => $obj->getUser() ? $obj->getUser()->getName() : "",
                'customer' => $obj->getCustomer() ? $obj->getCustomer()->getName() : "",
                'total' => $obj->getTotalCurrency()
            );
        }
        
        return new JsonModel($response);
        
    }

    /**
     * Orders list for bar chart
     * @return JsonModel
     */
    public function ordersListAction()
    {
        $beginDate = new \Commons\Util\DateTimeBr();
        $beginDate->add(\DateInterval::createFromDateString('-12 months'));
        $beginDate = DateTimeBr::createFromFormat('d-m-Y', '01-' . $beginDate->format('m-Y'));
        $endDate = new \Commons\Util\DateTimeBr();
        $endDate->modify("+1 day");
        
        $beginDateStr = $beginDate->format('Y-m-d');
        $endDateStr = $endDate->format('Y-m-d');
        
        if($this->identity()->isAdmin()){
            $sql = "select MONTH(orderDate) as month,COUNT(*) as qty, DATE_FORMAT(orderDate,'%m/%Y') as period, DATE_FORMAT(orderDate,'%m-%Y') as period from `order` where orderDate BETWEEN '{$beginDateStr}' AND '{$endDateStr}' GROUP BY 1";
        }else{
            $idUser = $this->identity()->getId();
            $sql = "select MONTH(orderDate) as month,COUNT(*) as qty, DATE_FORMAT(orderDate,'%m/%Y') as period, DATE_FORMAT(orderDate,'%m-%Y') as period from `order` where id_user = {$idUser} AND orderDate BETWEEN '{$beginDateStr}' AND '{$endDateStr}' GROUP BY 1";
        }
        
        $conn = $this->getEntityManager()->getConnection()->prepare($sql);
        $conn->execute();
        $results = $conn->fetchAll();
        
        $interval = \DateInterval::createFromDateString('1 month');
        $period = new \DatePeriod($beginDate, $interval, $endDate);
        
        $retorno = array();
        foreach ( $period as $dt ){

            $periodDateStr = $dt->format('m-Y');
            $total = 0;
            
            foreach($results as $obj){
                if($periodDateStr == $obj['period']){
                    $total = $obj['qty'];
                }
            }
            
            $retorno[] = array(
                'qty' => (int) $total,
                'period' => $periodDateStr,
                'day' => '01',
                'month' => $dt->format('m'),
                'year' => $dt->format('Y'),
                'label' => $dt->format('m/Y')
            );
            
        }
        
        
        return new JsonModel($retorno);
    }

    /**
     * Get notifications list
     * @return JsonModel
     */
    public function notificationsAction(){

        $retorno = array();

        if($this->identity()->isAdmin()){

            $devices = $this->getEntityManager()->getRepository('Model\\Entity\Device')->findBy(array('active' => false));
            if(count($devices) == 1){
                $retorno[] = array(
                    'message' => "Existe 1 dispositivo aguardando ativação.",
                    "link" => $this->url()->fromRoute('application/default', array('controller' => 'device', 'action' => 'index'))
                );
            }else if(count($devices) > 1){
                $retorno[] = array(
                    "message" => "Existem " . count($devices) . " dispositivos aguardando ativação.",
                    "link" => $this->url()->fromRoute('application/default', array('controller' => 'device', 'action' => 'index'))
                );
            }

        }

        return new JsonModel($retorno);
    }

    /**
     * Quick stats
     * @return JsonModel
     */
    public function quickStatsAction(){

        return new JsonModel(array(
            'customer'  => $this->getEntityManager()->createQuery("SELECT COUNT(ent.id) FROM Model\\Entity\\Customer ent")->getSingleScalarResult(),
            'product'   => $this->getEntityManager()->createQuery("SELECT COUNT(ent.id) FROM Model\\Entity\\Product ent")->getSingleScalarResult(),
            'order'     => $this->getEntityManager()->createQuery("SELECT COUNT(ent.id) FROM Model\\Entity\\Order ent")->getSingleScalarResult(),
            'device'    => $this->getEntityManager()->createQuery("SELECT COUNT(ent.id) FROM Model\\Entity\\Device ent")->getSingleScalarResult()
        ));

    }
}
