<?php
/**
 *
 * Abstract controller for api controllers
 *
 * @version 1
 * @author Renato Peterman <renato.pet (at) gmail.com>
 *
 */

namespace API\Controller;

use AP_XmlStrategy\View\Model\XmlModel;
use Exception;
use Zend\Mvc\Controller\AbstractActionController;
use Zend\Mvc\MvcEvent;
use Zend\View\Model\JsonModel;
use Zend\Http\Response\Stream;
use Zend\Http\Headers;
use Commons\Util\DateTimeBr;

abstract class AbstractApiActionController extends AbstractActionController
{
    protected $em;
    protected $config;
    protected $authManager;
    protected $whiteList = array('api-user', 'api-device', 'api-product-image', 'api-catalog-file', 'api-catalog-thumb');

    /**
     *
     * Return Config service
     *
     * @return ConfigService
     */
    public function getConfig()
    {
        if (is_array($this->config) === false)
            $this->config = $this->getServiceLocator()->get('Config');

        return $this->config;
    }

    /**
     * Get api logged user
     * @throws Exception
     */
    protected function getUser(){

        $token = $this->getRequestToken();
        if(!$token){
            throw new \Exception("User not logged");
        }

        $obj = $this->getEntityManager()->getRepository("Model\\Entity\\UserAccessToken")->findOneByAccessToken($token);
        if(!$obj){
            $obj = $this->getEntityManager()->getRepository("Model\\Entity\\OAuth2\\AccessToken")->findOneByAccessToken($token);
            if(!$obj){
                throw new \Exception("User not logged");
            }
        }

        return $obj->getUser();
    }

    /**
     * set Log data
     * @throws Exception
     */
    protected function setLog($code, $data)
    {
        $systemConfig = $this->getConfig();
        if (
            is_array($systemConfig) === false
            || isset($systemConfig['salestool']) === false
            || isset($systemConfig['salestool']['enableLog']) == false
            || $systemConfig['salestool']['enableLog'] !== true
        ) {
            return false;
        }

        if (empty($code))
            return false;

        try {
            $logEntity = new \Model\Entity\Log();
            $logEntity->setCode($code);
            $logEntity->setData($data);
            $logEntity->setSync(false);
            $logEntity->setSyncDatetime(new DateTimeBr());
            $this->getEntityManager()->persist($logEntity);
            $this->getEntityManager()->flush($logEntity);
            return true;
        } catch (Exception $ex) {
            //return $ex->getMessage();
            return false;
        }
    }

    /**
     * Return request token
     * @return string | boolean
     */
    protected function getRequestToken(){

        $data = $this->params()->fromQuery('access_token');
        if($data){
            return trim($data);
        }

        /* Post not allowed
        $post = $this->params()->fromPost('accessToken');
        if($post){
            return trim($post);
        }
        */

        if (!$this->getRequest()->getHeaders('Authorization')) {
            $authRedirect = isset($_SERVER['REDIRECT_HTTP_AUTHORIZATION']) ? $_SERVER['REDIRECT_HTTP_AUTHORIZATION'] : '';
            /*
            if (!isset($authRedirect) || !$authRedirect) {
                throw new ApiBadRequestException('Authorization error (1)');
            }
            */
            $authHeader = $authRedirect;
        } else {
            $authHeader = $this->getRequest()->getHeaders('Authorization')->getFieldValue();
        }

        if($authHeader){
            list($type,$token) = explode(' ', $authHeader);
            return $token;
        }

        return false;
    }

    /**
     * Authenticate application
     * @return boolean
     */
    protected function authenticate(){

        try{

            $token = $this->getRequestToken();
            if(!$token){
                throw new \Exception("Token not found");
            }

            $obj = $this->getEntityManager()->getRepository("Model\\Entity\\UserAccessToken")->findOneByAccessToken($token);
            if(!$obj){
                $obj = $this->getEntityManager()->getRepository("Model\\Entity\\OAuth2\\AccessToken")->findOneByAccessToken($token);
                if(!$obj){
                    throw new \Exception("Token not found");
                }
            }

            $currentDate = new \DateTime();
            if($currentDate > $obj->getExpires()){
                throw new \Exception("Token expired");
            }

            return true;

        }catch(Exception $ex){
            return false;
        }

        return false;

    }

    /**
     * Verify and authenticate Token before dispatch request
     * @override
     */
    public function onDispatch(MvcEvent $e) {

        if (
           !$this->authenticate()
           && !in_array($e->getRouteMatch()->getMatchedRouteName(), $this->whiteList)
           && !preg_match("/api\-custom\-[a-zA-Z0-9]*\-sync/", $e->getRouteMatch()->getMatchedRouteName())
        ){
            return $this->createCustomResponse($this->createUnauthorizedResponse()->serialize());
        }

        parent::onDispatch($e);
    }

    /* Services */

    /**
     * Doctrine entity manager
     * @return Doctrine\ORM\EntityManager
     */
    public function getEntityManager()
    {
        if (null === $this->em) {
            $this->em = $this->getServiceLocator()->get('Doctrine\ORM\EntityManager');
        }
        return $this->em;
    }

    public function removeAccent($string) {
        $remover = array("à" => "a","á" => "a","ã" => "a","â" => "a","é" => "e","ê" => "e","ì" => "i","í" => "i","ó" => "o","õ" => "o","ô" => "o","ú" => "u","ü" => "u","ç" => "c","À" => "A","Á" => "A","Ã" => "A","Â" => "A","É" => "E","Ê" => "E","Í" => "I","Ó" => "O","Õ" => "O","Ô" => "O","Ù" => "U","Ú" => "U","Ü" => "U","Ç" => "C");
        return strtr($string, $remover);
    }


    /* Responses */

    /**
     * Format the data that will be showed in view
     *
     * @param  array $data
     * @param  string $format
     * @return mixed
     */
    protected function createResponse($content, $httpStatusCode = 200, $error = false, $message = ''){

        $this->getResponseWithHeader()->setStatusCode($httpStatusCode);

        $data = null;
        if($error){
            $data = array('error' => $error, 'message' => $message);
        }else{
            $data = $content;
        }

        $format = $this->getEvent()->getRouteMatch()->getParam('format');

        if($format == 'xml'){
            return new XmlModel($data);
        }

        $zip = $this->params()->fromQuery('zip',false);

        if($zip){
            return $this->createCustomResponse(gzcompress(json_encode($data)));
        }else{
            return new JsonModel($data);
        }
    }

    protected function createErrorResponse($message, $status = 500){
        return $this->createResponse(null, $status, true, $message); // 200 - Http Request OK
    }

    protected function createBadRequestResponse(){
        return $this->createResponse(null, 400, true, "Bad request");
    }

    protected function createUnauthorizedResponse(){
        return $this->createResponse(null, 401, true, "Unauthorized request");
    }

    protected function createForbiddenResponse(){
        return $this->createResponse(null, 403, true, "Forbidden");
    }

    protected function createNotFoundResponse(){
        return $this->createResponse(null, 404, true, "Resource not found");
    }

    protected function createMethodNotAllowedResponse(){
        return $this->createResponse(null, 405, true, "Method not allowed");
    }


    /* Private */

    private function createCustomResponse($content, $gzip = false)
    {

        $format = $this->getEvent()->getRouteMatch()->getParam('format');
        $response = $this->getResponseWithHeader();
        if(!$format){
            $response->getHeaders()->addHeaderLine('Content-Type','application/json');
        }else{
            if($format == 'json'){
                $response->getHeaders()->addHeaderLine('Content-Type','application/json');
            }else if($format == 'xml'){
                $response->getHeaders()->addHeaderLine('Content-Type','application/xml');
            }
        }
        /*
        if($gzip){
            $response->getHeaders()->addHeaderLine('Content-Encoding','gzip');
        }
        */

        return $response->setContent($content);
    }

    private function getResponseWithHeader()
    {
        $response = $this->getResponse();
        $response->getHeaders()
                 ->addHeaderLine('Access-Control-Allow-Origin','*')
                 ->addHeaderLine('Access-Control-Allow-Methods','POST PUT DELETE GET');
        return $response;
    }

    public function createFileResponse($file, $contentType)
    {
        $response = new Stream();
        $response->setStream(fopen($file, 'r'));
        $response->setStatusCode(200);
        $headers = new Headers();
        $headers->addHeaderLine('Content-Transfer-Encoding', 'binary')
                ->addHeaderLine('Content-Type', $contentType)
                ->addHeaderLine('Content-Length', filesize($file));

        $response->setHeaders($headers);
        return $response;
    }

    public function getKeyValEntities($entities)
    {
        $response = array();
        if (count($entities) > 0) {
            foreach ($entities as $entity) {
                if (method_exists($entity, 'getId')) {
                    $response[$entity->getId()] = $entity;
                }
            }
        }
        return $response;
    }

    
    /* Atual */

    protected function getUrl() {

        $config = $this->getConfig();

        if (
            is_array($config)
            && isset($config['salestool'])
            && isset($config['salestool']['integration'])
            && isset($config['salestool']['integration']['config'])
            && isset($config['salestool']['integration']['config']['host'])
            && isset($config['salestool']['integration']['config']['port'])
        )
            return trim($config['salestool']['integration']['config']['host']).":".trim($config['salestool']['integration']['config']['port']);

        return false;
    }

    protected function get($apiServiceUrl, $decode = true, $timeout = 180)
    {
        try {

            $URL_API = $this->getUrl();
            if (empty($URL_API))
                throw new \Exception('O host do webservice não foi definido.');

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, str_replace(" ", '%20',$URL_API.$apiServiceUrl));
            curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: application/json', 'Authorization: Basic Q09ORVg6QzBOM1g=')); /*, 'Authorization: Basic '.$this->KEY_ID*/
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
            curl_setopt($ch, CURLOPT_HEADER, FALSE);
            curl_setopt($ch, CURLOPT_POST, FALSE);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, FALSE);
            curl_setopt($ch, CURLOPT_TIMEOUT, $timeout); //seconds (3min)

            $response = curl_exec($ch);

            curl_close($ch);

            if ($decode == true)
                $response = json_decode($response, true) ? $this->getLowercase(json_decode($response, true)) : false ;

            return $response;
        } catch(\Exception $ex) {
            return false;
        }
    }

    protected function post($apiServiceUrl, $fields)
    {
        try {

            $URL_API = $this->getUrl();
            if (empty($URL_API))
                throw new \Exception('O host do webservice não foi definido.');

            $fields = json_encode($fields);

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $URL_API.$apiServiceUrl);
            curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: application/json', 'Authorization: Basic Q09ORVg6QzBOM1g=')); /*, 'Authorization: Basic '.$this->KEY_ID*/
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
            curl_setopt($ch, CURLOPT_HEADER, FALSE);
            curl_setopt($ch, CURLOPT_POST, TRUE);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $fields);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, FALSE);

            $response = curl_exec($ch);
            curl_close($ch);

            if (strstr($response, 'cod_cadastro') !== false || strstr($response, 'id') !== false || strstr($response, 'chave_Fato') !== false)
                $response = json_decode($response, true) ? $this->getLowercase(json_decode($response, true)) : false;
            else
                $response = empty($response) ? true : false;

            return $response;
        } catch(\Exception $ex) {
            return false;
        }
    }

    protected function getLowercase($list)
    {
        $response = array();
        if (is_array($list) && count($list) > 0) {
            foreach ($list as $key => $val) {
                $response[trim(strtolower($key))] = $val;
            }
        }
        return $response;
    }

    
    protected function getDateDb($dateBr)
    {
        $myDateTime = \DateTime::createFromFormat('d/m/Y', $dateBr);
        return $myDateTime->format('Y-m-d');
    }
    
}
